<?php

namespace App\Http\Controllers\User;

use Illuminate\Http\Request;
use App\Models\Generalsetting;
use App\Http\Controllers\Controller;
use App\Traits\RespondsWithHttp;

class AuthorizationController extends Controller
{
    use RespondsWithHttp;

    public function twoStep(Request $request)
    {
        $user = auth()->user();
        if ($user->two_fa != 1) {
            return $this->respond($request, [], 'Two-step verification not required.', 'user.dashboard');
        }

        if ($this->expectsJson($request)) {
            return $this->respond($request, [], 'Two-step verification form not available via API.');
        }

        return view('user.twostep.otp_code');
    }

    public function twoStepVerify(Request $request)
    {
        $request->validate(['code' => 'required']);

        $user = auth()->user();
        if ($request->code != $user->two_fa_code) {
            return $this->respondWithError($request, 'Invalid OTP');
        }

        $user->two_fa = 0;
        $user->save();

        return $this->respond($request, [], 'Two-step verification successful.', 'user.dashboard');
    }

    public function twoStepResendCode(Request $request)
    {
        $code = randNum();
        $user = auth()->user();
        $user->two_fa_code = $code;
        $user->update();

        try {
            sendSMS($user->phone, 'Your two step authentication OTP is: ' . $code, Generalsetting::value('contact_no'));
        } catch (\Exception $e) {
            \Log::error('SMS sending failed: ' . $e->getMessage());
        }

        return $this->respond($request, [], 'OTP code is sent to your phone.', null);
    }

    public function verifyEmail(Request $request)
    {
        $user = auth()->user();
        if ($user->email_verified == 1) {
            return $this->respond($request, [], 'Email already verified.', 'user.dashboard');
        }

        try {
            @email([
                'email' => $user->email,
                'name' => $user->name,
                'subject' => __('Email Verification Code'),
                'message' => __('Email Verification Code is: ') . $user->verify_code,
            ]);
        } catch (\Exception $e) {
            \Log::error('Email sending failed: ' . $e->getMessage());
        }

        if ($this->expectsJson($request)) {
            return $this->respond($request, [], 'Verification code sent to your email.');
        }

        return view('user.email_verify.verification_code');
    }

    public function verifyEmailSubmit(Request $request)
    {
        $request->validate(['code' => 'required|integer']);

        $user = auth()->user();
        if (!$user) {
            return $this->respondWithError($request, 'User doesn\'t exist');
        }

        if ($user->verify_code != $request->code) {
            return $this->respondWithError($request, 'Invalid verification code.');
        }

        $user->verify_code = null;
        $user->email_verified = 1;
        $user->save();

        return $this->respond($request, [], 'Verification Successful, Welcome', 'user.dashboard');
    }

    public function verifyEmailResendCode(Request $request)
    {
        $code = randNum();
        $user = auth()->user();
        $user->verify_code = $code;
        $user->update();

        try {
            @email([
                'email' => $user->email,
                'name' => $user->name,
                'subject' => __('Email Verification Code'),
                'message' => __('Email Verification Code is: ') . $user->verify_code,
            ]);
        } catch (\Exception $e) {
            \Log::error('Email sending failed: ' . $e->getMessage());
        }

        return $this->respond($request, [], 'Verify code resent to your email.', null);
    }
}